// File: comm.h
//
// Copyright (c) 2007 by XXX
// -------------------------------------------------------------------------
// DESCRIPTION
// Header file containing OS specific function. 
//
// HISTORY
// Date         By who         Description
// -------------------------------------------------------------------------
// 04/26/2007   maddock     Header comments
// 05/12/2007   phyizal     Added fantom file support
// 05/16/2007   phyizal     Removed fantom function call 

#ifndef _COMM_H_
#define _COMM_H_

#include <stdlib.h>
#include <string.h>
#include <iostream>
#include "fantom/iNXT.h"
#include "fantom/iNXTIterator.h"
#include "fantom/tStatus.h"

//! Holds an array of 6 bluetooth adresses.
struct BTAddress { ViByte bluetoothAddress[6]; };
//! Holds 4 bytes that contain the signal strengths for the 4 bluetooth connections.
struct BTSignalStr { int connection[4]; };

//Error codes.
//TODO: Make them more comprehensive based on status code from NXT.
//       Also provide a method to translate numeric error code to more
//       user friendly error string.

enum ErrorCode
{		
	UnsupportedType = -1,
	IteratorFailed = -2,
	InvalidFileIterator = -3,
	IteratorAdvanceFailed = -4,
	InvalidFileHandler = -5,
	FileOpenFailed = -6,		
	WriteNotEnabled = -7,
	FileNotOpen = -8,		
	ReadFailed = -9,
	WriteFailed = -10,
	FileCloseFailed = -11,		
	WriteBufferIllegalSize = -12,
};

//! Abstraction for NXT iFile interface
struct NXTFile
{
	//! The NXTFile constructor.
	NXTFile() : file(NULL) , name(NULL)
	{
		size = 0;
		isOpen = false;
		isWriteable = false;
	}

	//! This constructs an NXTFile from the Fantom file type. Only in the Windows version.
	NXTFile(nFANTOM100::iFile* f): name((char*) malloc(20))
	{
      file = f;

	  if (f == NULL) return;

	  f->getName(name);

	  nFANTOM100::tStatus status;

	  size = f->getSize(status);

	  if (status.isFatal())
		  size = -1;

	  isOpen = false;
	  isWriteable =false;
	}

	//! The name of the file.
	char * name;
	//! The size in bytes of the file.
	int size;
	//! The mode the file is currently in.
	int mode;

	//! Whether the file is open or not.
	bool isOpen;
	//! Whether the file is writeable or not.
	bool isWriteable;

	//Won't be use in case of linux
	//! The Fantom file pointer. Only in the Windows version.
	nFANTOM100::iFile* file; 
	bool isValid();	
	int open(int mode, int size=100);
	int remove();
	int read(char* buff, int size);
	int write(char* buff, int size);
	int close();
};

//! Abstraction for NXT iFileIterator interface.
struct NXTFileIterator
{
	//! The NXTFileIterator constructor.
	NXTFileIterator() : iter(NULL)
	{
	}

	//! This constructs an NXTFileIterator from the Fantom file iterator type. Only in the Windows version.
	NXTFileIterator(nFANTOM100::iFileIterator *i)
	{
		iter = i;
	}

	//! The Fantom file iterator pointer. Only in the Windows version.
	nFANTOM100::iFileIterator *iter;
	//! Checks whether or not the NXTFileIterator is valid.
	bool isValid();
	//! Iterates to the next file in the NXTFileIterator.
	int next();
	//! Closes the NXTFileIterator.
	int close();
	//! Retrieves the current file into an NXTFile type.
	int getFile(NXTFile& file);
	//! Returns the size of the current file.
	int getCurrFileSize();
	//! Returns the name of the current file.
	const char* getCurrFileName();
	
};

//! Abstraction for NXT iModule interface.
struct NXTModule
{
	//! The NXTModule constructor.
	NXTModule() : internalModule(NULL)
	{
	}

	//! This constructs an NXTModule from the Fantom module type. Only in the Windows version.
	NXTModule(nFANTOM100::iModule *m)
	{
		internalModule = m;
	}

	//! The Fantom module pointer. Only in the Windows version.
	nFANTOM100::iModule *internalModule;
	//! Retrieves the name of the module.
	int getName(std::string &name);
	//! Retrieves the ID of the module.
	int getModuleID(int &id);
	//! Retrieves the size of the module.
	int getModuleSize(int &size);
	//! Retrieves the IOMap size.
	int getIOMapSize(int &size);
	//! Reads the IOMap into a buffer.
	int readIOMap(ViPByte buffer, int offset, int number_of_bytes);
	//! Writes data to the IOMap.
	int writeIOMap(const ViPByte, int offset, int number_of_bytes);

};

//! Abstraction for NXT iModuleIterator interface.
struct NXTModuleIterator
{
	//! The NXTModuleIterator
	NXTModuleIterator() : internalIter(NULL)
	{
	}

	//! This constructs an NXTModule from the Fantom module type. Only in the Windows version.
	NXTModuleIterator(nFANTOM100::iModuleIterator *i)
	{
		internalIter = i;
	}

	//! The Fantom module iterator pointer. Only in the Windows version.
	nFANTOM100::iModuleIterator *internalIter;
	//! Iterates to the next module in the NXTModuleIterator.
	int next();
	//! Retrieves the current module into an NXTModule type.
	int getModule(NXTModule &module);
	//! Returns the name of the current module.
	int getName(std::string &s);
};

//! Checks to see if an NXT file is valid.
bool NXTFile::isValid()
{
	return file != NULL;
}

//! Opens an NXT file.
int NXTFile::open(int mode, int size)
{	
	bool iswrite = false;

	if (!isValid())
		return InvalidFileHandler;

	nFANTOM100::tStatus status;

	switch(mode)
	{
	case 0:
		//read
		file->openForRead(status);
		break;

	case 1:
		//write
		file->openForWrite(size, status);
		iswrite = true;
		break;

	case 2:
		//append
		file->openForDataAppend(status);
		iswrite = true;
		break;

	case 3:
		//dataWrite
		file->openForDataWrite(size, status);
		iswrite = true;
		break;

	case 4:
		file->openForLinearWrite(size, status);
		iswrite = true;
		break;

	default:
		return UnsupportedType;

	}

	if (status.isNotFatal())
	{
		isOpen = true;
		isWriteable = iswrite;
		return 0;
	}

	isOpen = false;
	isWriteable = false;

	return FileOpenFailed;
}

//! Deletes a file from the NXT.
int NXTFile::remove() 
{
     if (!isValid())
		 return InvalidFileHandler;

	 nFANTOM100::tStatus status;
	 file->remove(status);

	 if (status.isNotFatal())
		 return 0;

	 return -1;

}

//! Reads the file into a buffer.
int NXTFile::read(char *buff, int size) 
{
	if (!isValid())
		return InvalidFileHandler;

	if (!isOpen)
		return FileNotOpen;

	nFANTOM100::tStatus status;
	file->read(reinterpret_cast<ViPBuf> (buff), size, status);

	if (status.isNotFatal())
		return 0;

	return ReadFailed;

}

//! Writes data to the file.
int NXTFile::write(char *buff, int size)
{
	if (!isValid())
		return InvalidFileHandler;

	if (!isOpen)
		return FileNotOpen;

	if (!isWriteable)
		return WriteNotEnabled;

	if (size < 0)
		return WriteBufferIllegalSize;

	nFANTOM100::tStatus status;
    file->write(reinterpret_cast<ViPBuf> (buff), size, status);

	if (status.isNotFatal())
		return 0;

	return WriteFailed;
}

//! Closes the NXTFile.
int NXTFile::close()
{
	if (!isValid())
		return 0;

	nFANTOM100::tStatus status;
	file->close(status);

	return 0;
}

const char* NXTFileIterator::getCurrFileName()
{
	if (!isValid())
		return NULL;

	nFANTOM100::tStatus status;
	char *name = (char*) malloc(20);
	
	iter->getName(name, status);	

	if (status.isNotFatal())
		return name;

	return NULL;
}

int NXTFileIterator::getCurrFileSize()
{
	if (!isValid())
		return InvalidFileIterator;

	nFANTOM100::tStatus status;
	int size = iter->getSize(status);

	if (status.isNotFatal())
		return size;

	return -1;
}

int NXTFileIterator::getFile(NXTFile& file)
{
	if (!isValid())
		return InvalidFileIterator;

	nFANTOM100::tStatus status;
	nFANTOM100::iFile *fptr = iter->getFile(status);

	if (status.isNotFatal())
		file = NXTFile(fptr);
    else
		return -1;

	return 0;
}

int NXTFileIterator::next()
{
	if (!isValid())
		return InvalidFileIterator;

	nFANTOM100::tStatus status;
	iter->advance(status);

	if (status.isNotFatal()) 
		return 0;

	return IteratorAdvanceFailed;
}

bool NXTFileIterator::isValid()
{
	return iter != NULL;
}

int NXTModuleIterator::next()
{
	int res = 0;
	nFANTOM100::tStatus status;
	if (internalIter != NULL)
	{
		internalIter->advance(status);
		if(status.isNotFatal())
			res = 0;
		else
			res = -1;
	}
	else res = -1;

	return res;
}

int NXTModuleIterator::getModule(NXTModule &module)
{
	if (internalIter == NULL)
		return -1;

	int res = 0;
	nFANTOM100::tStatus status;
	nFANTOM100::iModule *m = internalIter->getModule(status);

	if (status.isNotFatal())
	    module = NXTModule(m);
	else
		res = -1;

	return res;
}

int NXTModuleIterator::getName(std::string &s)
{
	if (internalIter == NULL)
		return -1;

	int res = 0;
	nFANTOM100::tStatus status;
	char *name = (char *)malloc(20);
	internalIter->getName(name, status);

	if (status.isNotFatal())
	{
	    s = name;
		
	}
	else
		res = status.getCode();

	return res;
}

int NXTModule::getName(std::string &s)
{
	if (internalModule == NULL)
		return -1;

	int res = 0;
	//nFANTOM100::tStatus status;
	char *name = (char *)malloc(20);
	internalModule->getName(name);

	/*if (status.isNotFatal())
	    s = name;
	else
		res = -1;*/

	return res;
}

int NXTModule::getModuleID(int &id)
{
	if (internalModule == NULL)
		return -1;

	int res = 0;
	//nFANTOM100::tStatus status;	
	id = internalModule->getModuleID();

	/*if (status.isFatal())
	    res = -1;*/

	return res;
}

int NXTModule::getModuleSize(int &size)
{
	if (internalModule == NULL)
		return -1;

	int res = 0;
	//nFANTOM100::tStatus status;	
	size = internalModule->getModuleSize();

	/*if (status.isFatal())
	    res = -1;*/

	return res;
}

int NXTModule::getIOMapSize(int &size)
{
	if (internalModule == NULL)
		return -1;

	int res = 0;
	//nFANTOM100::tStatus status;	
	size = internalModule->getModuleIOMapSize();

	/*if (status.isFatal())
	    res = -1;*/

	return res;
}

int NXTModule::readIOMap(ViPByte buffer, int offset, int bytes)
{
	if (internalModule == NULL)
		return -1;

	int res = 0;
	nFANTOM100::tStatus status;	
	internalModule->readIOMap(offset, bytes, buffer, status);

	if (status.isFatal())
	    res = -1;

	return res;
}

int NXTModule::writeIOMap(ViPByte buffer, int offset, int bytes)
{
	if (internalModule == NULL)
		return -1;

	int res = 0;
	nFANTOM100::tStatus status;	
	internalModule->writeIOMap(offset, bytes, buffer, status);

	if (status.isFatal())
	    res = -1;

	return res;
}

bool Open();
bool OpenBT();
void Close();
void SendDirectCommand(ViBoolean response, ViByte * dc_buf, int dc_buf_size, ViByte *re_buf, int re_buf_size);
double GetProtocolVersion();
double GetFirmwareVersion();
std::string GetName();
BTAddress GetBluetoothAddress();
BTSignalStr GetSignalStrength();
double GetAvailableFlash();
//bool Write();

nFANTOM100::tStatus status;
nFANTOM100::iNXTIterator* nxtIteratorPtr = NULL;
nFANTOM100::iNXT* nxtPtr = NULL;
//For fetching list of files
nFANTOM100::iFileIterator* fileIteratorPtr = NULL;

//! NXT++ communication helper functions. Basic users should not need to use this.
namespace Comm
{
	bool Open()
	{
		// Create an NXT iterator object which is used to find all accessible NXT devices.
		nxtIteratorPtr = nFANTOM100::iNXT::createNXTIterator(
			false /* don't search for NXTs over Bluetooth (only search USB) */,
			1 /* timeout for Bluetooth discovery ignored */, status );
	
		// Creating the NXT iterator object could fail, better check status before dereferencing a
		//    potentially NULL pointer.
		if( status.isNotFatal())
		{
			// Create an NXT object for the first NXT that was found.  Note that if a NXT is found
			//    over BT, the computer and the NXT must be paired before an NXT object can be
			//    created.  This can be done programatically using the iNXT::pairBluetooth method.
			nxtPtr = nxtIteratorPtr->getNXT( status );
	
			// Destroy the NXT iterator object which we no longer need
			nFANTOM100::iNXT::destroyNXTIterator( nxtIteratorPtr );
		}
		return status.isNotFatal();
	}
	
	bool OpenBT()
	{
		// Create an NXT iterator object which is used to find all accessible NXT devices.
		nxtIteratorPtr = nFANTOM100::iNXT::createNXTIterator(
			true /* don't search for NXTs over Bluetooth (only search USB) */,
			1 /* timeout for Bluetooth discovery ignored */, status );
				
		// Creating the NXT iterator object could fail, better check status before dereferencing a
		// potentially NULL pointer.
		if( status.isNotFatal())
		{
			// Create an NXT object for the first NXT that was found.  Note that if a NXT is found
			//    over BT, the computer and the NXT must be paired before an NXT object can be
			//    created.  This can be done programatically using the iNXT::pairBluetooth method.
			nxtPtr = nxtIteratorPtr->getNXT( status );
	
			// Destroy the NXT iterator object which we no longer need
			nFANTOM100::iNXT::destroyNXTIterator( nxtIteratorPtr );
		}
		return status.isNotFatal();
	}
	
	void Close()
	{
		// Destroy the NXT object.
		nFANTOM100::iNXT::destroyNXT( nxtPtr );
	}
	
	void SendDirectCommand(ViBoolean response, ViByte * dc_buf, int dc_buf_size, ViByte *re_buf, int re_buf_size) 
	{
		nxtPtr->sendDirectCommand( response /* a response is not required for this direct command */,
			dc_buf, dc_buf_size,
			re_buf /* no response buffer */, re_buf_size /* no response buffer, specify 0 for size */, status );
	}
	
	// Why is this in here? It's not refrenced by anything
	/* bool Write()
	{
		ViUInt8 sysCommandBuffer[] = {0x01, 0x88};
		
		/*, 0x2A, 0x2E, 0x72, 0x78, 0x65, 0x00, 0x00,0x00,0x00,0x00,
			0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};*//*
        
		nxtPtr->write(reinterpret_cast<ViByte*> (sysCommandBuffer), sizeof (sysCommandBuffer), status);

		
		ViUInt8 sysCommandResponse[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
		nxtPtr->read(reinterpret_cast<ViPBuf> (sysCommandResponse), 7, status);


		std::cout<<(int)sysCommandResponse[3]<<std::endl;

		return status.isFatal();
	} */

	double GetProtocolVersion()
	{
		ViUInt8 protocolVersionMajor, protocolVersionMinor, firmwareVersionMajor, firmwareVersionMinor;
		nxtPtr->getFirmwareVersion( protocolVersionMajor, protocolVersionMinor, firmwareVersionMajor, firmwareVersionMinor, status );
		double version = protocolVersionMinor;
		while(version >= 1)
			version /= 10;
		version += protocolVersionMajor;
		return version;
	}
	
	double GetFirmwareVersion()
	{
		ViUInt8 protocolVersionMajor, protocolVersionMinor, firmwareVersionMajor, firmwareVersionMinor;
		nxtPtr->getFirmwareVersion( protocolVersionMajor, protocolVersionMinor, firmwareVersionMajor, firmwareVersionMinor, status );
		double version = firmwareVersionMinor;
		while(version >= 1)
			version /= 10;
		version += firmwareVersionMajor;
		return version;
	}
	
	std::string GetName()
	{
		ViChar name[16];
		ViByte bluetoothAddress[6];
		ViUInt8 signalStrength[4];
		ViUInt32 availableFlash;
		nxtPtr->getDeviceInfo( name, bluetoothAddress, signalStrength, availableFlash, status );
		name[15] = NULL;
		return (std::string)name;
	}

	BTAddress GetBluetoothAddress()
	{
		ViChar name[16];
		ViByte bluetoothAddress[6];
		ViUInt8 signalStrength[4];
		ViUInt32 availableFlash;
		nxtPtr->getDeviceInfo( name, bluetoothAddress, signalStrength, availableFlash, status );
		BTAddress address;
		for(int i = 0; i < 6; i++)
		{
			address.bluetoothAddress[i] = bluetoothAddress[i];
		}
		return address;
	}

	BTSignalStr GetSignalStrength()
	{
		ViChar name[16];
		ViByte bluetoothAddress[6];
		ViUInt8 signalStrength[4];
		ViUInt32 availableFlash;
		nxtPtr->getDeviceInfo( name, bluetoothAddress, signalStrength, availableFlash, status );
		BTSignalStr strength;
		strength.connection[0] = signalStrength[0];
		strength.connection[1] = signalStrength[1];
		strength.connection[2] = signalStrength[2];
		strength.connection[3] = signalStrength[3];
		return strength;
	}

	//! Returns the amount of flash memory left
	double GetAvailableFlash()
	{
		ViChar name[16];
		ViByte bluetoothAddress[6];
		ViUInt8 signalStrength[4];
		ViUInt32 availableFlash;
		nxtPtr->getDeviceInfo( name, bluetoothAddress, signalStrength, availableFlash, status );
		return (double)availableFlash/1024;
	}

	//! Retrieves a specified file into an NXTFile.
	NXTFile GetNXTFile(std::string filename, bool &isFatal)
	{
		isFatal = true;
		nFANTOM100::tStatus status;

		NXTFile file;		
		nFANTOM100::iFile* iFile = nxtPtr->createFile(filename.c_str(), status);

		if (status.isNotFatal())
		{
			file = NXTFile(iFile);
			isFatal = false;
		}

		return file;

	}

	//! Searches for files on the NXT and stores the results in an NXTFileIterator.
	NXTFileIterator GetFileIterator(std::string pattern, bool &isFatal) 
	{
		isFatal = true;
		nFANTOM100::tStatus status;

		NXTFileIterator iter;		
		nFANTOM100::iFileIterator* fileI = nxtPtr->createFileIterator(pattern.c_str(), status);
		
		if (status.isNotFatal())
		{		    
		    iter.iter = fileI;
			isFatal = false;
		}		

		return iter;
	}

	//!Destroys an NXTFileIterator.
	void destroyFileIterator(NXTFileIterator &iter)
	{
		if (iter.isValid())
			nxtPtr->destroyFileIterator(iter.iter);
	}

	//! Destroys an NXTFile.
	void destroyFile(NXTFile &file)
	{
		if (file.isValid())
			nxtPtr->destroyFile(file.file);
	}

	//! Searches for modules on the NXT and stores the results in an NXTModuleIterator.
	NXTModuleIterator GetModuleIterator(std::string pattern, int &statusCode)
	{
		statusCode = 0;
		nFANTOM100::tStatus stat;

		NXTModuleIterator iter;		
		nFANTOM100::iModuleIterator* moduleI = nxtPtr->createModuleIterator(pattern.c_str(), stat);

		if (stat.isNotFatal())
			iter = NXTModuleIterator (moduleI);
		else
			statusCode = stat.getCode();	
		return iter;
	}

	//! Destroys an NXTModuleIterator.
	void destroyModuleIterator(NXTModuleIterator &iter)
	{
		nxtPtr->destroyModuleIterator(iter.internalIter);
	}

	//! Retrieves a specified module into an NXTModule.
	NXTModule GetModule(std::string moduleName, int moduleID, int moduleSize, int ioMapSize, int &statCode)
	{
		statCode = 0;
		nFANTOM100::tStatus status;

		NXTModule internalMod;
		char *mname = const_cast<char*> (moduleName.data());
		nFANTOM100::iModule* module = 
			nxtPtr->createModule(mname, moduleID, moduleSize, ioMapSize, status);

		if (status.isNotFatal())
			internalMod.internalModule = module;
		else
			statCode = status.getCode();

		return internalMod;
	}

	//! Destroys an NXTModule.
	void destroyModule(NXTModule &module)
	{
		nxtPtr->destroyModule(module.internalModule);
	}
}

#endif
